<?php
/* File name:   vfcg-lib.php
 * File URI:    http://victorfont.com/reusable-custom-function-lib
 * Description: This file is a collection of custom functions, filters, and hooks we use in every Genesis theme.
 * Usage:       Copy this into your child theme's /lib/ directory and add include_once( get_stylesheet_directory() . '/lib/vfcg-lib.php' ); to functions.php
 * Version:     1.0
 * Author:      Victor M. Font Jr.
 * Author URI:  http://victorfont.com/
 * License:     GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

/* !0. TABLE OF CONTENTS */

/*

	1. Keep users logged in for 1 month
	2. Remove page title for a specific page(s)
	3. Restore HTML to Content Limit Excerpts
	4. URL Shortcode
	5. No Self Pings
	6. JPG Image Quality
	7. Schema.org Website Markup
    8. Schema.org itemprop=image Markup
    9. Menu Descriptions (Walker)
   10. get_theme_version()
   11. get_asset_current_version_number( $asset_file )
   12. site_is_in_debug_mode()
   13. change_theme_stylesheet_uri_to_min_version( $stylesheet_uri, $stylesheet_dir_uri )
   14. iPhone X support
   15. disable adsense auto ads in Genesis 2.6+
   16. Create a shortcode to display custom Go to top link
   17. move comments below comment form
   18. disable WordPress emojis
   19. Append directives to the virtual robots.txt (edit yourdomain.com in code below)
   20. Modify Genesis SItemap Output
*/

/*  !1. Keep users logged in for 1 month */
add_filter( 'auth_cookie_expiration', 'keep_me_logged_in_for_1_month' );

/**
 * keep_me_logged_in_for_1_month function.
 *
 * @access public
 * @param mixed $expirein
 * @return 1 month in seconds
 */
function keep_me_logged_in_for_1_month( $expirein ) {
    return 2592000; // 1 month in seconds
}

/*  !2. Remove page title for a specific page(s) (requires HTML5 theme support) */
/* Change '28' to your page id */
/**
 * child_remove_page_titles function.
 *
 * @access public
 * @return void
 */
add_action( 'get_header', 'child_remove_page_titles' );
function child_remove_page_titles() {

    //* 354 =
    //* 519 = Leadership Framework
    //* 1149 = Training Registration
    //* 3545 = Resume/CV
    //* 4544 = Welcome Back
    //* 1754 = Plugin Defect Form
    //* 5199 = project inquiry

    $pages = array( );
    if ( is_page( $pages ) ) {
        remove_action( 'genesis_entry_header', 'genesis_do_post_title' );
    }
}

/*  !3. Restore HTML to Content Limit Excerpts */
add_filter( 'get_the_content_limit_allowedtags', 'get_the_content_limit_custom_allowedtags' );
/**
 * @author Brad Dalton
 * @example http://wp.me/p1lTu0-a5w
 */
function get_the_content_limit_custom_allowedtags() {
    // Add custom tags to this string
    return '<script>,<style>,<span>,<br>,<em>,<i>,<ul>,<ol>,<li>,<a>';
}

/*  !4. URL Shortcode */
/*  @author Bill Erickson
 * @link http://www.billerickson.net/wordpress-shortcode-sidebar-widget/
 * @return Site URL.
 */
function url_shortcode() {
    return get_bloginfo('url');
}
add_shortcode('url','url_shortcode');

/*  !5. No Self Pings */
//Pass the variable by reference to the function, so the function can modify the variable.
function no_self_ping (&$links) {
    $home = get_option( 'home' );
    foreach ( $links as $l => $link ) {
        //Find the position of the first occurrence of a substring in a string.
        //($a === $b) Identical operator. TRUE if $a is equal to $b, and they are of the same type.
        if ( 0 === strpos( $link, $home ) ) {
            //Unset the variable
            unset($links[$l]);
        }
    }
}
// Hooks the function to the specific action (pre_ping)
add_action( 'pre_ping', 'no_self_ping' );

/* !6. JPG Image Quality */
add_filter( 'jpeg_quality', create_function( '', 'return 100;' ) );
add_filter( 'wp_editor_set_quality', create_function( '', 'return 100;' ) );

/* !7. Schema.org Website Markup */
add_filter('genesis_attr_head', 'vmf_head_attributes');
function vmf_head_attributes( $attributes ) {
    $attributes['itemscope'] = 'itemscope';
    $attributes['itemtype'] = 'http://schema.org/WebSite';

    return $attributes;
}

/* !8. Schema.org itemprop=image Markup */
add_filter('the_content', 'vmf_add_itemprop_image_markup', 2);
function vmf_add_itemprop_image_markup($content)
{
    //Replace the instance with the itemprop image markup.
    $string = '<img';
    $replace = '<img itemprop="image"';
    $content = str_replace( $string, $replace, $content );
    return $content;
}

/* !9. Menu Descriptions */
add_filter( 'walker_nav_menu_start_el', 'be_add_description', 10, 2 );
function be_add_description( $item_output, $item ) {
	$description = $item->post_content;
	if ( ' ' !== $description )
		return preg_replace( '/(<a.*?>[^<]*?)</', '$1' . '<span class="menu-description">' . $description . '</span><', $item_output);
	else
	return $item_output;
}

/* Functions from Tonya Mork, https://knowthecode.io
--------------------------------------------- */

if ( ! function_exists( 'get_theme_version' ) ) :
	/** !10.
	 * Get the theme's version.
	 *
	 * When in development/mode mode, it uses the style.css modification time.
	 * Else, it grabs the version number from the stylesheet.
	 *
	 * @since 1.0.0
	 *
	 * @return string|int
	 */
	function get_theme_version() {
		if ( site_is_in_debug_mode() ) {
			return get_asset_current_version_number(
				get_stylesheet_directory() . '/style.css'
			);
		}
		$theme = wp_get_theme();
		return $theme->get( 'Version' );
	}
endif;

if ( ! function_exists( 'get_asset_current_version_number' ) ) :
	/** !11.
	 * Get the specified asset file's current version number.
	 * This function gets the file's modification time when
	 * the site is in development/debug mode.
	 *
	 * @since 1.0.0
	 *
	 * @param $asset_file
	 *
	 * @return bool|int
	 */
	function get_asset_current_version_number( $asset_file ) {
		return filemtime( $asset_file );
	}
endif;

if ( ! function_exists( 'site_is_in_debug_mode' ) ) :
	/** !12.
	 * Checks if the site is in development/debug mode.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	function site_is_in_debug_mode() {
		if ( ! defined( 'SCRIPT_DEBUG' ) ) {
			return false;
		}
		return ( (bool) SCRIPT_DEBUG === true );
	}
endif;

if ( ! function_exists( 'change_theme_stylesheet_uri_to_min_version' ) ) :
	add_filter( 'stylesheet_uri', 'change_theme_stylesheet_uri_to_min_version', 9999, 2 );
	/** !13.
	 * Change the theme's stylesheet URI to minified version when not
	 * in development/debug mode.
	 *
	 * @since 1.0.0
	 *
	 * @param string $stylesheet_uri Stylesheet URI for the current theme/child theme.
	 * @param string $stylesheet_dir_uri Stylesheet directory URI for the current theme/child theme.
	 *
	 * @return string
	 */
	function change_theme_stylesheet_uri_to_min_version( $stylesheet_uri, $stylesheet_dir_uri ) {
		if ( site_is_in_debug_mode() ) {
			return $stylesheet_uri;
		}
		$minified_stylesheet_filename = '/style.min.css';
		if ( ! file_exists( get_stylesheet_directory() . $minified_stylesheet_filename ) ) {
			return $stylesheet_uri;
		}
		return $stylesheet_dir_uri . $minified_stylesheet_filename;
	}
endif;
// end - Tony Mork functions

/* !14. adjust viewport for iPhone X camera notch - must include related CSS below in style.css body tag */
add_filter( 'genesis_viewport_value', 'iphonex_viewport' );
function iphonex_viewport( $content ){
    return "width=device-width, initial-scale=1, viewport-fit=cover";
}

/* Mobile Safari only - iPhone X notch support */
/* uncomment and include in style.css body tag */
/* padding: constant(safe-area-inset-top) constant(safe-area-inset-right) constant(safe-area-inset-bottom) constant(safe-area-inset-left); */

/* !15. disable adsense auto ads in Genesis 2.6+
 * to restore adsense, comment out the to add_filters and add_action below */

if ( defined('PARENT_THEME_BRANCH') && PARENT_THEME_BRANCH >= '2.6') {
    // Sets AdSense ID to always be an empty string - stops meta box from appearing on Post screens.
    add_filter( 'genesis_pre_get_option_adsense_id', '__return_empty_string' );
    // Removes AdSense metabox from Theme Settings.
    add_action( 'genesis_theme_settings_metaboxes', function () {
    	remove_meta_box( 'genesis-theme-settings-adsense', 'toplevel_page_genesis', 'main' );
    });
    // Removes AdSense ID setting from Customizer.
    add_filter( 'genesis_customizer_theme_settings_config', function ( $config ) {
    	unset( $config['genesis']['sections']['genesis_adsense'] );
    	return $config;
    });
}

//* !16. Create a shortcode to display our custom Go to top link  */
add_shortcode('footer_custombacktotop', 'set_footer_custombacktotop');
function set_footer_custombacktotop($atts) {
	return '<a class="button gototop" href="#" aria-label="Return to top of page"><i class="far fa-arrow-alt-circle-up"><span class="screen-reader-text">Return to top of page</span></i></a>';
}

//* Add smooth scrolling for any link having the class of "top"
add_action('wp_footer', 'go_to_top');

function go_to_top() { ?>
	<script type="text/javascript">
		jQuery(function($) {
			$('a.button.gototop').click(function() {
				$('html, body').animate({scrollTop:0}, 'slow');
			return false;
			});
		});
	</script>
<?php }

/** !17. move comments below comment form */

add_action( 'genesis_before_comments' , 'move_comments_below_form' );
function move_comments_below_form()
{
    if ( is_single() && have_comments() ) {

        remove_action( 'genesis_comment_form', 'genesis_do_comment_form' );

        add_action( 'genesis_comments', 'genesis_do_comment_form', 5 );
    }
}

/** !18. disable WordPress emojis */

function disable_wp_emojicons() {

  // all actions related to emojis
  remove_action( 'admin_print_styles', 'print_emoji_styles' );
  remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
  remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
  remove_action( 'wp_print_styles', 'print_emoji_styles' );
  remove_filter( 'wp_mail', 'wp_staticize_emoji_for_email' );
  remove_filter( 'the_content_feed', 'wp_staticize_emoji' );
  remove_filter( 'comment_text_rss', 'wp_staticize_emoji' );

  // filter to remove TinyMCE emojis
  add_filter( 'tiny_mce_plugins', 'disable_emojicons_tinymce' );
}
// add_action( 'init', 'disable_wp_emojicons' );

function disable_emojicons_tinymce( $plugins ) {
  if ( is_array( $plugins ) ) {
    return array_diff( $plugins, array( 'wpemoji' ) );
  } else {
    return array();
  }
}

// add_filter( 'emoji_svg_url', '__return_false' );

/* !19. Append directives to the virtual robots.txt
    Siteground adds crawl-delay: 10 to robots.txt. Don't use physical robots.txt on SiteGround
    edit yourdomaim.com below and remove comment tag before add_filter
*/
// add_filter( 'robots_txt', 'robots_mod', 10, 2 );
function robots_mod( $output, $public ) {
    $output .= "Sitemap: https://yourdomain.com/sitemap_index.xml";
    return $output;
}

/* !20. modify Genesis sitemap output */
add_filter( 'genesis_sitemap_output', 'vmf_sitemap' );
function vmf_sitemap() {

	$heading = ( genesis_a11y( 'headings' ) ? 'h2' : 'h4' );

	$sitemap  =  sprintf( '<%2$s>%1$s</%2$s>', __( 'Categories:', 'genesis' ) , $heading );
	$sitemap .=  sprintf( '<ul>%s</ul>', wp_list_categories( 'sort_column=name&title_li=&echo=0' ) );
/*
	$sitemap .=  sprintf( '<%2$s>%1$s</%2$s>', __( 'Authors:', 'genesis' ) , $heading );
	$sitemap .=  sprintf( '<ul>%s</ul>', wp_list_authors( 'exclude_admin=0&optioncount=1&echo=0' ) );

	$sitemap .=  sprintf( '<%2$s>%1$s</%2$s>', __( 'Monthly:', 'genesis' ) , $heading );
	$sitemap .=  sprintf( '<ul>%s</ul>', wp_get_archives( 'type=monthly&echo=0' ) );
*/
	$sitemap .=  sprintf( '<%2$s>%1$s</%2$s>', __( 'Recent Posts:', 'genesis' ) , $heading );
	$sitemap .=  sprintf( '<ul>%s</ul>', wp_get_archives( 'type=postbypost&limit=100&echo=0' ) );

    $sitemap .=  sprintf( '<%2$s>%1$s</%2$s>', __( 'Pages:', 'genesis' ), $heading );
	$sitemap .=  sprintf( '<ul>%s</ul>', wp_list_pages( 'title_li=&echo=0' ) );

	return $sitemap;
}